/*
  ==============================================================================

    SonicCrypt Chaos Engine
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#include "PluginProcessor.h"
#include "PluginEditor.h"

SonicCryptChaosAudioProcessorEditor::SonicCryptChaosAudioProcessorEditor(SonicCryptChaosAudioProcessor& p)
    : AudioProcessorEditor(&p), audioProcessor(p)
{
    setResizable(true, true); setResizeLimits(800, 550, 1920, 1080);

    // UPDATED: Use stored size from processor
    setSize(audioProcessor.lastUIWidth, audioProcessor.lastUIHeight);

    if (BinaryData::logo_pngSize > 0) logoImage = juce::ImageCache::getFromMemory(BinaryData::logo_png, BinaryData::logo_pngSize);

    // SIDEBAR LEFT
    if (logoImage.isValid()) logoButton.setImages(false, true, true, logoImage, 1.0f, juce::Colours::transparentBlack, logoImage, 1.0f, juce::Colours::white.withAlpha(0.2f), logoImage, 1.0f, juce::Colours::transparentBlack);
    logoButton.onClick = [] { juce::URL("https://soniccrypt.org").launchInDefaultBrowser(); }; addAndMakeVisible(logoButton);
    addAndMakeVisible(copyrightBtn); copyrightBtn.setButtonText(juce::CharPointer_UTF8("\xc2\xa9 2025 SonicCrypt"));
    copyrightBtn.setColour(juce::TextButton::buttonColourId, juce::Colours::transparentBlack); copyrightBtn.setColour(juce::TextButton::textColourOffId, juce::Colours::grey.withAlpha(0.6f));
    copyrightBtn.onClick = [this] { aboutOverlay.setVisible(true); aboutOverlay.toFront(true); };

    addAndMakeVisible(presetGroup); presetGroup.setColour(juce::GroupComponent::outlineColourId, juce::Colours::darkgrey); presetGroup.setColour(juce::GroupComponent::textColourId, juce::Colours::white);
    addAndMakeVisible(presetListBox); presetListBox.setRowHeight(25); presetListBox.setColour(juce::ListBox::backgroundColourId, juce::Colours::black.withAlpha(0.3f));
    presetListModel = std::make_unique<PresetListComponent>(presetListBox, [this](juce::File file) {
        overlay.setVisible(true); overlay.toFront(true);
        juce::Timer::callAfterDelay(50, [this, file]() { audioProcessor.loadPresetFile(file); overlay.setVisible(false);
        juce::File f = audioProcessor.getCurrentSampleFolder(); folderPathLabel.setText(f.isDirectory() ? f.getFileName() : "Preset Loaded", juce::dontSendNotification);
            });
        });
    presetListBox.setModel(presetListModel.get()); presetListModel->refresh(audioProcessor.getPresetFolder());

    addAndMakeVisible(savePresetButton); savePresetButton.addListener(this); savePresetButton.setColour(juce::TextButton::buttonColourId, juce::Colour::fromString("FF00BFFF")); savePresetButton.setColour(juce::TextButton::textColourOffId, juce::Colours::black);
    addAndMakeVisible(refreshButton); refreshButton.addListener(this); refreshButton.setColour(juce::TextButton::buttonColourId, juce::Colours::darkgrey);

    // HEADER RIGHT
    addAndMakeVisible(loadFolderButton); loadFolderButton.addListener(this); loadFolderButton.setColour(juce::TextButton::buttonColourId, juce::Colours::darkgrey);
    addAndMakeVisible(folderPathLabel); juce::File f = audioProcessor.getCurrentSampleFolder();
    folderPathLabel.setText(f.isDirectory() ? f.getFileName() : "No Folder Loaded", juce::dontSendNotification);
    folderPathLabel.setColour(juce::Label::textColourId, juce::Colours::grey); folderPathLabel.setJustificationType(juce::Justification::centred);

    // CENTER UI - CHAOS ORB
    addAndMakeVisible(chaosButton);
    chaosButton.onClick = [this]() {
        overlay.setVisible(true); overlay.toFront(true);
        juce::Timer::callAfterDelay(50, [this]() { audioProcessor.triggerRandomChaos(); overlay.setVisible(false); });
        };

    addAndMakeVisible(typeLabel); typeLabel.setText("TYPE", juce::dontSendNotification); typeLabel.setColour(juce::Label::textColourId, juce::Colours::grey); typeLabel.setJustificationType(juce::Justification::centred); typeLabel.setFont(juce::FontOptions(12.0f).withStyle("Bold"));
    addAndMakeVisible(modeSelector); modeSelector.addItem("One Shot", 1); modeSelector.addItem("Rhythm", 2); modeSelector.addItem("Drone", 3);
    modeAttachment = std::make_unique<juce::AudioProcessorValueTreeState::ComboBoxAttachment>(audioProcessor.getAPVTS(), "mode", modeSelector);

    addAndMakeVisible(rhythmRateLabel); rhythmRateLabel.setText("GRID", juce::dontSendNotification); rhythmRateLabel.setColour(juce::Label::textColourId, juce::Colours::grey); rhythmRateLabel.setFont(juce::FontOptions(12.0f)); rhythmRateLabel.setJustificationType(juce::Justification::centred);
    addAndMakeVisible(rhythmRateSelector); rhythmRateSelector.addItem("1/4", 1); rhythmRateSelector.addItem("1/8", 2); rhythmRateSelector.addItem("1/16", 3); rhythmRateSelector.addItem("1/32", 4);
    rhythmRateAttachment = std::make_unique<juce::AudioProcessorValueTreeState::ComboBoxAttachment>(audioProcessor.getAPVTS(), "rhythmRate", rhythmRateSelector);

    // LFO
    addAndMakeVisible(lfoGroup); lfoGroup.setText("DRIFT LFO"); lfoGroup.setColour(juce::GroupComponent::outlineColourId, juce::Colours::grey); lfoGroup.setColour(juce::GroupComponent::textColourId, juce::Colour::fromString("FF00BFFF"));
    addAndMakeVisible(lfoActiveButton); lfoActiveAttachment = std::make_unique<juce::AudioProcessorValueTreeState::ButtonAttachment>(audioProcessor.getAPVTS(), "lfoOn", lfoActiveButton);
    addAndMakeVisible(lfoRateSelector); lfoRateSelector.addItem("1/1", 1); lfoRateSelector.addItem("1/2", 2); lfoRateSelector.addItem("1/4", 3); lfoRateSelector.addItem("1/8", 4); lfoRateSelector.addItem("1/16", 5); lfoRateSelector.addItem("1/32", 6);
    lfoRateAttachment = std::make_unique<juce::AudioProcessorValueTreeState::ComboBoxAttachment>(audioProcessor.getAPVTS(), "lfoRate", lfoRateSelector);
    addAndMakeVisible(lfoRateLabel); lfoRateLabel.setText("Rate", juce::dontSendNotification); lfoRateLabel.setFont(juce::FontOptions(12.0f));
    addAndMakeVisible(lfoDepthSlider); lfoDepthSlider.setSliderStyle(juce::Slider::RotaryHorizontalVerticalDrag); lfoDepthSlider.setTextBoxStyle(juce::Slider::NoTextBox, false, 0, 0); lfoDepthSlider.setColour(juce::Slider::rotarySliderFillColourId, juce::Colours::white);
    lfoDepthAttachment = std::make_unique<juce::AudioProcessorValueTreeState::SliderAttachment>(audioProcessor.getAPVTS(), "lfoDepth", lfoDepthSlider);
    addAndMakeVisible(lfoDepthLabel); lfoDepthLabel.setText("Depth", juce::dontSendNotification); lfoDepthLabel.setFont(juce::FontOptions(12.0f)); lfoDepthLabel.setJustificationType(juce::Justification::centred);

    setupSlider(driveSlider, driveLabel, driveAttachment, "drive", "DIRT"); setupSlider(filterSlider, filterLabel, filterAttachment, "cutoff", "CUTOFF");
    setupSlider(resSlider, resLabel, resAttachment, "resonance", "RES"); setupSlider(reverbSlider, reverbLabel, reverbAttachment, "reverbMix", "SPACE");
    setupSlider(delaySlider, delayLabel, delayAttachment, "delayTime", "ECHO");

    addAndMakeVisible(overlay); overlay.setVisible(false); addAndMakeVisible(aboutOverlay); aboutOverlay.setVisible(false);
    startTimerHz(30);

    constructorFinished = true;
}

SonicCryptChaosAudioProcessorEditor::~SonicCryptChaosAudioProcessorEditor() { stopTimer(); }

void SonicCryptChaosAudioProcessorEditor::setupSlider(juce::Slider& s, juce::Label& l, std::unique_ptr<SliderAttachment>& att, juce::String id, juce::String txt) {
    addAndMakeVisible(s); addAndMakeVisible(l); l.setText(txt, juce::dontSendNotification);
    s.setSliderStyle(juce::Slider::RotaryHorizontalVerticalDrag); s.setTextBoxStyle(juce::Slider::NoTextBox, false, 0, 0);
    s.setColour(juce::Slider::rotarySliderFillColourId, juce::Colour::fromString("FF00BFFF")); s.setColour(juce::Slider::thumbColourId, juce::Colours::white);
    l.setJustificationType(juce::Justification::centred); l.setFont(juce::FontOptions(14.0f).withStyle("Bold"));
    att = std::make_unique<SliderAttachment>(audioProcessor.getAPVTS(), id, s);
}
void SonicCryptChaosAudioProcessorEditor::showAboutWindow() {}

void SonicCryptChaosAudioProcessorEditor::paint(juce::Graphics& g) {
    g.fillAll(juce::Colour::fromString("FF121212")); g.setColour(juce::Colour::fromString("FF1E1E1E"));
    float s = (float)getWidth() / BASE_WIDTH;
    int sidebarW = 230 * s;

    g.fillRoundedRectangle(10.0f * s, 10.0f * s, (float)sidebarW, (float)getHeight() - 20.0f * s, 5.0f);

    float mainX = 10.0f * s + sidebarW + 10.0f * s;
    float mainW = getWidth() - mainX - 10.0f * s;
    g.fillRoundedRectangle(mainX, 10.0f * s, mainW, 80.0f * s, 5.0f);

    float bodyY = 100.0f * s;
    g.fillRoundedRectangle(mainX, bodyY, mainW, (float)getHeight() - bodyY - 10.0f * s, 5.0f);

    g.setColour(juce::Colours::grey); g.setFont(juce::FontOptions(14.0f * s));
    g.drawText("CHAOS ENGINE v1.0", mainX + mainW - 160 * s, 10.0f * s, 150 * s, 80 * s, juce::Justification::centredRight);

    if (overlay.isVisible()) overlay.repaint();
}

void SonicCryptChaosAudioProcessorEditor::resized() {
    // FIX: Größe sofort speichern, wenn sie valide ist
    if (constructorFinished) {
        // FIX: Größe sofort speichern, wenn sie valide ist
        if (getWidth() > 100 && getHeight() > 100) {
            audioProcessor.lastUIWidth = getWidth();
            audioProcessor.lastUIHeight = getHeight();

            // Auch direkt in den ValueTree schreiben für maximale Sicherheit
            audioProcessor.getAPVTS().state.setProperty("uiWidth", getWidth(), nullptr);
            audioProcessor.getAPVTS().state.setProperty("uiHeight", getHeight(), nullptr);
        }
    }

    float s = (float)getWidth() / BASE_WIDTH;
    int sidebarW = 230 * s;
    int margin = 10 * s;

    // --- SIDEBAR (LEFT) ---
    logoButton.setBounds(margin + 5 * s, margin + 5 * s, sidebarW - 10 * s, 60 * s);
    copyrightBtn.setBounds(margin + 5 * s, margin + 70 * s, sidebarW - 10 * s, 20 * s);

    int presetsY = 110 * s;
    presetGroup.setBounds(margin, presetsY, sidebarW, getHeight() - presetsY - 50 * s);
    presetListBox.setBounds(margin + 5 * s, presetsY + 20 * s, sidebarW - 10 * s, presetGroup.getHeight() - 30 * s);

    int btnY = getHeight() - 40 * s;
    int btnW = (sidebarW - 10 * s) / 2;
    savePresetButton.setBounds(margin, btnY, btnW - 2 * s, 30 * s);
    refreshButton.setBounds(margin + btnW + 2 * s, btnY, btnW - 2 * s, 30 * s);

    // --- HEADER (RIGHT) ---
    int mainX = margin + sidebarW + margin;
    int mainW = getWidth() - mainX - margin;

    int headerH = 80 * s;
    int headerCenterY = 10 * s + headerH / 2;
    int ctrlH = 30 * s;

    loadFolderButton.setBounds(mainX + 20 * s, headerCenterY - ctrlH / 2, 120 * s, ctrlH);
    folderPathLabel.setBounds(mainX + 150 * s, headerCenterY - ctrlH / 2, 300 * s, ctrlH);

    // --- CENTER (MAIN) ---
    int bodyY = 100 * s;
    int bodyH = getHeight() - bodyY - margin;
    int cx = mainX + mainW / 2;
    int cy = bodyY + bodyH / 2 - 40 * s;

    int chaosSize = 160 * s;
    chaosButton.setBounds(cx - chaosSize / 2, cy - chaosSize / 2, chaosSize, chaosSize);

    int leftX = cx - chaosSize / 2 - 150 * s;
    modeSelector.setBounds(leftX, cy - 15 * s, 120 * s, 30 * s);
    typeLabel.setBounds(leftX, cy - 35 * s, 120 * s, 20 * s);

    rhythmRateSelector.setBounds(leftX + 20 * s, cy + 45 * s, 80 * s, 20 * s);
    rhythmRateLabel.setBounds(leftX + 20 * s, cy + 25 * s, 80 * s, 20 * s);

    int rightX = cx + chaosSize / 2 + 30 * s;
    lfoGroup.setBounds(rightX, cy - 70 * s, 160 * s, 140 * s);

    int gx = lfoGroup.getX(); int gy = lfoGroup.getY();
    lfoActiveButton.setBounds(gx + 10 * s, gy + 25 * s, 60 * s, 20 * s);
    lfoDepthSlider.setBounds(gx + 80 * s, gy + 25 * s, 70 * s, 70 * s);
    lfoDepthLabel.setBounds(gx + 80 * s, gy + 90 * s, 70 * s, 20 * s);

    lfoRateLabel.setBounds(gx + 10 * s, gy + 115 * s, 40 * s, 20 * s);
    lfoRateSelector.setBounds(gx + 55 * s, gy + 115 * s, 90 * s, 20 * s);

    int knobsY = cy + 110 * s;
    int ks = 80 * s; int g = 30 * s;
    int totalKW = 5 * ks + 4 * g; int sx = mainX + (mainW - totalKW) / 2;
    auto lk = [&](juce::Slider& sl, juce::Label& lb, int i) { sl.setBounds(sx + i * (ks + g), knobsY, ks, ks); lb.setBounds(sx + i * (ks + g), knobsY - 20 * s, ks, 20 * s); };
    lk(driveSlider, driveLabel, 0); lk(filterSlider, filterLabel, 1); lk(resSlider, resLabel, 2);
    lk(reverbSlider, reverbLabel, 3); lk(delaySlider, delayLabel, 4);

    overlay.setBounds(getLocalBounds()); aboutOverlay.setBounds(getLocalBounds());
}

void SonicCryptChaosAudioProcessorEditor::timerCallback() {
    if (audioProcessor.isLoading) repaint();

    chaosButton.setLevel(audioProcessor.currentOutputLevel);

    int m = *audioProcessor.getAPVTS().getRawParameterValue("mode");
    rhythmRateSelector.setEnabled(m == 1); rhythmRateLabel.setAlpha(m == 1 ? 1.0f : 0.4f);
}

void SonicCryptChaosAudioProcessorEditor::buttonClicked(juce::Button* button) {
    if (button == &loadFolderButton) { fileChooser = std::make_shared<juce::FileChooser>("Select Samples", juce::File::getSpecialLocation(juce::File::userMusicDirectory)); fileChooser->launchAsync(juce::FileBrowserComponent::openMode | juce::FileBrowserComponent::canSelectDirectories, [this](const juce::FileChooser& fc) { auto f = fc.getResult(); if (f.exists()) { folderPathLabel.setText(f.getFileName(), juce::dontSendNotification); overlay.setVisible(true); overlay.toFront(true); juce::Timer::callAfterDelay(50, [this, f]() { audioProcessor.loadSamplesFromFolder(f); overlay.setVisible(false); }); } }); }
    else if (button == &savePresetButton) { auto d = audioProcessor.getPresetFolder(); fileChooser = std::make_shared<juce::FileChooser>("Save Preset", d, "*.scp"); fileChooser->launchAsync(juce::FileBrowserComponent::saveMode | juce::FileBrowserComponent::canSelectFiles, [this](const juce::FileChooser& fc) { auto f = fc.getResult(); if (f != juce::File{}) { if (!f.hasFileExtension("scp")) f = f.withFileExtension("scp"); audioProcessor.savePreset(f.getFileNameWithoutExtension()); presetListModel->refresh(audioProcessor.getPresetFolder()); } }); }
    else if (button == &refreshButton) { presetListModel->refresh(audioProcessor.getPresetFolder()); }
}